/******************************************************************************
*                            Freescale Semiconductor
*
* Name: Main_App.c
*
* Description: Main application routines
*
* MCU: Kinetis (K5x)
*
* Version for IAR 6.1
*******************************************************************************/

/******************************************************************************
*                                  Includes
******************************************************************************/
#include "Main_App.h"

//#include "MCU.h"        /* MCU Specific */
//#include "USB_CDC.h"    /* USB_CDC Support */

/******************************************************************************
*                                 Definitions
******************************************************************************/

#if DEBUG_MODE==FALSE
//#define SEND_SINE_WAVE

#define CONNECTED_TO_USB	1         //PTADD_PTADD0 == 1	//Change this
#define CONNECTED_TO_LCD_BOARD	0	  //IRQSC_IRQF == 1
#define GUI_TOLD_ME_TO_GO_STAND_ALONE	0 //	PTADD_PTADD2	//change this

/******************************************************************************
*                                   Structs
******************************************************************************/

typedef union 
{
	byte Byte;
	struct 
	{
		byte ConnectedToUsb      	:1;
		byte ConnectedToLcdBoard	:1;
		byte ZigbeeEnabled		:1;
		byte BluetoothEnabled		:1;
		byte :1;
		byte :1;
		byte :1;
		byte :1;
	} Bits;
} SystemFlags;


typedef enum
{
	NO_MEASUREMENT,
	GLU_MEASUREMENT,
	BPM_MEASUREMENT,
	BPM_LEAK_TEST,
	ECG_MEASUREMENT,
	SPR_MEASUREMENT,
	HEIGHT_MEASUREMENT,
	WEIGHT_MEASUREMENT,
	TEMPERATURE_MEASUREMENT,
        POX_MEASUREMENT
};

/******************************************************************************
*                                   Locals
******************************************************************************/
/* Virtual COM Application start Init Flag */
static volatile boolean start_app = FALSE;
/* Virtual COM Application Carrier Activate Flag */
static volatile boolean start_transactions = FALSE;
/* Receive Buffer */
static uint_8 g_curr_recv_buf[DATA_BUFF_SIZE];
/* Send Buffer */
static uint_8 g_curr_send_buf[DATA_BUFF_SIZE];
/* Receive Data Size */
static uint_8 g_recv_size;
/* Send Data Size */
static uint_8 g_send_size;



UINT8 InBuffer[32];	//From the PC to the MCU
UINT8 OutBuffer[128];	//From the MCU to the PC
UINT8 OutSize;
UINT8 InSize;

UINT16 IdNumber = 0;
UINT8 MainActualState;
UINT8 ActualMeasurement;

UINT8 TimerSendDummyData;

void TimerSendDummyData_Event(void);
void EnableAFE(UINT8 measurementId);
void vfnEnable_AFE (void);


typedef enum
{
	MAIN_STATE_DISCONNECTED,
	MAIN_STATE_CONTROLLED_BY_LCD_BOARD,
	MAIN_STATE_CONTROLLED_BY_USB
} MainStates_e;

/*****************************************************************************
*                             Function definitions
******************************************************************************/

static void USB_App_Callback(uint_8 controller_ID,
                        uint_8 event_type, void* val);
static void USB_Notify_Callback(uint_8 controller_ID,
                        uint_8 event_type, void* val);

/* Applications*/
void TestApp_Init(void);
void TestApp_Task(void);

/* States */
void StateDisconnected(void);
void StateControlledByLcdBoard(void);
void StateControlledByUsb(void);

/* AFEs functions */

/* GlucoseMeter */
void GluStartMeasurementReq(void);
void GluAbortMeasurementReq(void);
void GluStartCalibrationReq(void);
void GlucometerBloodDetectedInd(void);
void GlucometerMeasurementOkInd(void);
void GlucometerCalibrationOkInd(void);
void GlucometerDebugNewDataReadyInd(void);


/* Bpm */
void BpmStartMeasurementReq(void);
void BpmAbortMeasurementReq(void);
void BpmMeasurementCompleteInd(void);
void BpmMeasurementErrorInd(void);
void BpmNewDataReadyInd(void);	//for debug only
void BmpSendPressureValue(void);


void BpmStartLeakTestReq(void);
void BpmAbortLeakTestReq(void);
void BpmLeakTestCompleteInd(void);

/* Ecg */
void EcgHeartRateStartMeasurementReq(void);
void EcgHeartRateAbortMeasurementReq(void);
void EcgHeartRateMeasurementCompleteOkInd(void);
void EcgHeartRateMeasurementErrorInd(void);
void EcgHeartBeatOccurredInd(void);

void EcgDiagnosticModeStartMeasurementReq(void);
void EcgDiagnosticModeStopMeasurementReq(void);
void EcgDiagnosticModeNewDataReadyInd(void);

/* Hwt */
void HwtReadTemperatureReq(void);
void HwtReadHeightReq(void);
void HwtReadWeightReq(void);
void HwtTemperatureMeasurmentCompleteInd(void);
void HwtHeightMeasurementCompleteInd(void);
void HwtWeightMeasurementCompleteInd(void);


/* Spr */
void SprDiagnosticModeStartMeasurementReq(void);
void SprDiagnosticModeStopMeasurementReq(void);
void SprDiagnosticModeNewDataReadyInd(void);


/* Pox */
void PoxStartMeasurementReq(void);
void PoxAbortMeasurementReq(void);
void PoxMeasurementCompleteOkInd(void);
void PoxMeasurementErrorInd(void);


void PoxDiagnosticModeStartMeasurementReq(void);
void PoxDiagnosticModeStopMeasurementReq(void);
void PoxNewDataReadyInd(void);
void PoxDebugModeNewDataReadyInd(void);

/* system restart */
void SystemRestartReq(void);

/* Spr 2nd pass */
const pFunc_t Spr_Events[] = 
{
	NULL,//EVENT_SPR_NONE,
	NULL,//EVENT_SPR_MEASUREMENT_COMPLETE_OK,
	NULL,//EVENT_SPR_MEASUREMENT_ERROR,
	SprDiagnosticModeNewDataReadyInd	//EVENT_SPR_NEW_DATA_READY
};


/* Serial comm */
void SerialComm_PeriodicTask(void);
void SerialComm_SendData(void);



/** Pointer to function that will be used as a callback */
/** Pointer to function that will be used as a callback */
const pFunc_t GlucoseMeter_Events[] = 
{
	NULL,	//EVENT_GLU_NONE,
	GlucometerBloodDetectedInd,	//EVENT_GLU_BLOOD_DETECTED,
	GlucometerMeasurementOkInd,	//EVENT_GLU_FINISH_MEASUREMENT_OK,
	GlucometerCalibrationOkInd,	//EVENT_GLU_FINISH_CALIBRATION_OK
	GlucometerDebugNewDataReadyInd,			//EVENT_GLU_DEBUG_NEW_DATA_READY
};


const pFunc_t EcgDsc_Events[] = 
{
	NULL,									//EVENT_ECG_NONE,
	EcgHeartRateMeasurementCompleteOkInd,	//EVENT_ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	EcgHeartRateMeasurementErrorInd,		//EVENT_ECG_HEART_RATE_MEASUREMENT_ERROR,
	EcgHeartBeatOccurredInd,				//EVENT_ECG_HEART_BEAT_OCCURRED,
	EcgDiagnosticModeNewDataReadyInd		//EVENT_ECG_DIAGNOSTIC_MODE_NEW_DATA_READY	
};

const pFunc_t Ecg_Events[] = 
{
	NULL,									//EVENT_ECG_NONE,
	EcgHeartRateMeasurementCompleteOkInd,	//EVENT_ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	EcgHeartRateMeasurementErrorInd,		//EVENT_ECG_HEART_RATE_MEASUREMENT_ERROR,
	NULL,	//EcgHeartBeatOccurredInd,				//EVENT_ECG_HEART_BEAT_OCCURRED,
	EcgDiagnosticModeNewDataReadyInd		//EVENT_ECG_DIAGNOSTIC_MODE_NEW_DATA_READY	
};


const pFunc_t Bpm_Events[] = 
{
	NULL,						//EVENT_BPM_NONE,
	BpmMeasurementCompleteInd,	//EVENT_BPM_MEASUREMENT_COMPLETE_OK,
	BpmMeasurementErrorInd,		//EVENT_BPM_MEASUREMENT_ERROR,
	BpmLeakTestCompleteInd,		//EVENT_BPM_LEAK_TEST_COMPLETE,
	BpmNewDataReadyInd,			//EVENT_BPM_NEW_DATA_READY	//for debug only
	BmpSendPressureValue		//EVENT_BPM_SEND_PRESSURE_VALUE_TO_PC
};


const pFunc_t Pox_Events[] = 
{
	NULL,							//POX_EVENT_NONE,
	PoxMeasurementCompleteOkInd, 	//POX_EVENT_MEASUREMENT_COMPLETE_OK,
	PoxMeasurementErrorInd,			//POX_EVENT_MEASUREMENT_ERROR,
	PoxNewDataReadyInd,				//POX_EVENT_NEW_DATA_READY
	PoxDebugModeNewDataReadyInd		//POX_EVENT_DEBUG_MODE_NEW_DATA_READY
};


const pFunc_t Hwt_Events[] = 
{
	NULL,		//EVENT_HWT_NONE,	
	HwtHeightMeasurementCompleteInd,		//EVENT_HEIGHT_MEASUREMENT_ERROR, //Send the height withh value of 0
	HwtHeightMeasurementCompleteInd,		//EVENT_HEIGHT_MEASUREMENT_COMPLETE_OK,
	HwtWeightMeasurementCompleteInd,		//EVENT_WEIGHT_MEASUREMENT_COMPLETE_OK,
	HwtTemperatureMeasurmentCompleteInd		//EVENT_TEMPERATURE_MEASUREMENT_COMPLETE_OK		
};


const pFunc_t MainStateMachine[] = 
{
	StateDisconnected,
	StateControlledByLcdBoard,
	StateControlledByUsb
};



const pFunc_t ExecuteCommandReq[] =
{
	GluStartMeasurementReq,				//GLU_START_MEASUREMENT,
	GluAbortMeasurementReq,				//GLU_ABORT_MEASUREMENT,
	GluStartCalibrationReq,				//GLU_START_CALIBRATION,
	NULL,								//GLU_BLOOD_DETECTED,
	NULL,								//GLU_MEASUREMENT_COMPLETE_OK,
	NULL,								//GLU_CALIBRATION_COMPLETE_OK,
	
	BpmStartMeasurementReq,				//BPM_START_MEASUREMENT,
	BpmAbortMeasurementReq,				//BPM_ABORT_MEASUREMENT,
	NULL,								//BPM_MEASUREMENT_COMPLETE_OK,
	NULL,								//BPM_MEASUREMENT_ERROR,
										  
	BpmStartLeakTestReq,				//BPM_START_LEAK_TEST,
	BpmAbortLeakTestReq,					//BPM_ABORT_LEAK_TEST,
	NULL,								//BPM_LEAK_TEST_COMPLETE,	
	
	EcgHeartRateStartMeasurementReq,	//ECG_HEART_RATE_START_MEASUREMENT,
	EcgHeartRateAbortMeasurementReq,	//ECG_HEART_RATE_ABORT_MEASUREMENT,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_COMPLETE_OK,
	NULL,								//ECG_HEART_RATE_MEASUREMENT_ERROR,
	NULL,								//ECG_HEART_BEAT_OCCURRED,
	
	EcgDiagnosticModeStartMeasurementReq,//ECG_DIAGNOSTIC_MODE_START_MEASUREMENT,
	EcgDiagnosticModeStopMeasurementReq, //ECG_DIAGNOSTIC_MODE_STOP_MEASUREMENT,
	NULL,								 //ECG_DIAGNOSTIC_MODE_NEW_DATA_READY,	
		
	HwtReadTemperatureReq,				//TMP_READ_TEMPERATURE,
	HwtReadHeightReq,					//HGT_READ_HEIGHT,
	HwtReadWeightReq,					//WGT_READ_WEIGHT,

	NULL,	//SprStartMeasurementReq,				//SPR_START_MEASUREMENT,
	NULL,	//SprAbortMeasurementReq,				//SPR_ABORT_MEASUREMENT,
	NULL,								//SPR_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_MEASUREMENT_ERROR,

	SprDiagnosticModeStartMeasurementReq,								//SPR_DIAGNOSTIC_MODE_START_MEASURMENT,
	SprDiagnosticModeStopMeasurementReq,								//SPR_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//SPR_DIAGNOSTIC_MODE_NEW_DATA_READY,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_COMPLETE_OK,
	NULL,								//SPR_DIAGNOSTIC_MODE_MEASUREMENT_ERROR,
		
	PoxStartMeasurementReq,				//POX_START_MEASUREMENT,
	PoxAbortMeasurementReq,				//POX_ABORT_MEASURMENT,
	NULL,								//POX_MEASURMENT_COMPLETE_OK,
	NULL,								//POX_MEASURMENT_ERROR,

	PoxDiagnosticModeStartMeasurementReq,//POX_DIAGNOSTIC_MODE_START_MEASURMENT,
	PoxDiagnosticModeStopMeasurementReq,//POX_DIAGNOSTIC_MODE_STOP_MEASURMENT,
	NULL,								//POX_DIAGNOSTIC_MODE_NEW_DATA_READY
	NULL,	//BPM_SEND_PRESSURE_VALUE_TO_PC,
	SystemRestartReq,	//SYSTEM_RESTART,
	NULL	//BPM_DATA_READY = 0xFF,
};

/*******************************************************************************
*                                Functions
********************************************************************************/



void TestApp_Init(void) 
{       
    uint_8   error; 

    vfnEnable_AFE ();   
    SwTimer_Init();

#ifdef SEND_SINE_WAVE
    TimerSendDummyData.msCount = 64;
    TimerSendDummyData.pfnTimerCallback = TimerSendDummyData_Event;
#endif
    /*******************************/
     
    g_recv_size = 0;
    g_send_size= 0;
    DisableInterrupts;		
    #if (defined _MCF51MM256_H) || (defined _MCF51JE256_H)
     usb_int_dis();
    #endif
    /* Initialize the USB interface */
    error = USB_Class_CDC_Init(CONTROLLER_ID,USB_App_Callback,
                                NULL,USB_Notify_Callback);
    if(error != USB_OK)
    {
        /* Error initializing USB-CDC Class */
        return;
    }
    EnableInterrupts;
	#if (defined _MCF51MM256_H) || (defined _MCF51JE256_H)
     usb_int_en();
    #endif

}

void TestApp_Task(void)
{  
  GlucoseMeter_PeriodicTask();      
		
		
  SwTimer_PeriodicTask();
  SerialComm_PeriodicTask();
  MainStateMachine[MainActualState](); 
}


static const UINT16 SinX[] = 
{
	1250, 1349, 1448, 1545, 1640, 1733, 1823, 1910, 1992, 2069, 2142, 2208, 2269, 2323, 2371, 2411, 2444, 2470, 2488, 2498, 2500, 2494, 2481, 2459, 2430, 2394, 2350, 2300, 2243, 2179, 2110, 2036, 1956, 1872, 1784, 1693, 1598, 1502, 1404, 1305, 1206, 1107, 1009, 912, 818, 726, 638, 553, 473, 398, 328, 264, 206, 155, 111, 73, 44, 21, 7, 0, 2, 11, 28, 53, 85, 124, 171, 225, 284, 350, 422, 499, 581, 667, 756, 849, 944, 1041, 1140, 1239
};

#define SIN_X_LAST_ELEMENT 	79
volatile UINT8 EcgBufferSize = 64;
volatile UINT8 Time	= 64;


void TimerSendDummyData_Event(void)
{
	if (ActualMeasurement == ECG_MEASUREMENT)
	{		
		static UINT8 SinXArrayActualElement = 0;

		//Send indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = ECG_DIAGNOSTIC_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = EcgBufferSize + 2;				//data bytes
		
		OutBuffer[OutSize++] = (UINT8) (IdNumber >> 8);
		OutBuffer[OutSize++] = (UINT8) (IdNumber & 0x00FF);
		
		IdNumber++;
		
		
		while (OutSize < (EcgBufferSize + DATA_PACKET + 2))
		{
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] >> 8;		
			OutBuffer[OutSize++] = SinX[SinXArrayActualElement] & 0x00FF;
			
			if (SinXArrayActualElement == SIN_X_LAST_ELEMENT)
			{
				SinXArrayActualElement = 0;
			}
			else
			{
				SinXArrayActualElement++;
			}
		}

		(void)SerialComm_SendData();
		
		SwTimer_StartTimer(TimerSendDummyData, Time);
	}	
	
}

void StateDisconnected(void)
{
	if (CONNECTED_TO_USB)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
		else if (CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
	}	
}


void StateControlledByLcdBoard(void)
{
	if (CONNECTED_TO_USB)
	{
		//Send LL16 to sleep
		//add code here
		MainActualState = MAIN_STATE_CONTROLLED_BY_USB;
	}
	else if (!CONNECTED_TO_LCD_BOARD)
	{
		MainActualState = MAIN_STATE_DISCONNECTED;
	}
	
	//run SPI process here for communication with LL16 
}



void StateControlledByUsb(void)
{
	if (CONNECTED_TO_LCD_BOARD)
	{
		if (!CONNECTED_TO_USB || GUI_TOLD_ME_TO_GO_STAND_ALONE)
		{
			//wake up LL16
			//add code here
			MainActualState = MAIN_STATE_CONTROLLED_BY_LCD_BOARD;
		}
	}
	else //!CONNECTED_TO_LCD_BOARD
	{
		if (!CONNECTED_TO_USB)
		{
			MainActualState = MAIN_STATE_DISCONNECTED;	
		}		
	}
		
	//CDC_Engine();	//run CDC task old usb stack delete later
}





void SerialComm_PeriodicTask(void)
{	
  unsigned char u8BufferCounter;
  
	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{			
		//if there is data, read it
		if(g_recv_size)
		{
                 for (u8BufferCounter=0; u8BufferCounter<g_recv_size; u8BufferCounter++)
                 {
                   InBuffer[u8BufferCounter]=g_curr_recv_buf[u8BufferCounter]; 
                 }
		 InSize = g_recv_size;
                 g_recv_size=0;
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//check if there is new data from the LCD board
		//add SPI receive function here	
	}


	//if there is data, parse it and execute command
	if (InSize>0)
	{	
		//there is new data
		if (InBuffer[PACKET_TYPE] == REQ)
		{
			if (ExecuteCommandReq[InBuffer[COMMAND_OPCODE]] != NULL)
			{
				//check if OPCCODE is in a valid range
				if ((InBuffer[COMMAND_OPCODE] <= LAST_COMMAND))
				{
					ExecuteCommandReq[InBuffer[COMMAND_OPCODE]]();
				}
			}	
		}
		//else
		//{
			//packet type is not a request
		//}
		InSize = 0;
	}
}



void SerialComm_SendData(void)
{
	UINT8 i=0;
        UINT8 u8Status=0;

	if (MainActualState == MAIN_STATE_CONTROLLED_BY_USB)
	{			
		if (OutSize>0)			
		{			
			g_send_size = OutSize;
                        for(i=0;i<OutSize;i++)
                        {
                         g_curr_send_buf[i]=OutBuffer[i]; 
                        }
                        
                        u8Status = USB_Class_CDC_Interface_DIC_Send_Data(CONTROLLER_ID, &g_curr_send_buf[0],g_send_size);
		
			OutSize = 0;
		}
	}
	else if (MainActualState == MAIN_STATE_CONTROLLED_BY_LCD_BOARD)
	{
		//add SPI send function here	
	}	
}


#define MEDICAL_SENSOR_BOARD_IIC_ADDRESS		0x03

const UINT8 CmdEnableGlu	=	0x11;	
const UINT8 CmdEnableSpr	=	0x22;	
const UINT8 CmdEnablePox	=	0x33;	
const UINT8 CmdEnableEcg	=	0x44;	
const UINT8 CmdEnableBpm	=	0x55;	
const UINT8 CmdEnableHwt	=	0x66;	
const UINT8 CmdDisableAll	=	0xFF;



void EnableAFE(UINT8 measurementId)
{
	UINT8 delay=0xFF;
	
	AFE_DISABLE();		//power off board
	
	while(delay--)	//wait a few us
	{
		;
	}
	
	IIC_DISABLE();
	vfnIICV1Init();
	AFE_ENABLE();		//power on board
	
	switch (measurementId)		//send IIC data to select device
	{
		case NO_MEASUREMENT:
			break;
			
		case GLU_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableGlu, 1);
			break;
			
		case BPM_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableBpm, 1);
			break;
						
		case ECG_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableEcg, 1);
			break;
		
		case SPR_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableSpr, 1);
			break;
			
		case HEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableHwt, 1);
			break;		
		
		case WEIGHT_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableHwt, 1);
			break;		
		
		case TEMPERATURE_MEASUREMENT:
			vfnIICV1Write(MEDICAL_SENSOR_BOARD_IIC_ADDRESS, (UINT8*)&CmdEnableHwt, 1);
			break;		
	}	
}


void SystemRestartReq(void)
{
	//stop all measurements
	GlucoseMeter_AbortMeasurement();

}

/**************************************************
		Glucose Meter
******************************************************/


void GluStartMeasurementReq(void)
{
	if (ActualMeasurement == NO_MEASUREMENT)
	{
		vfnStartGLUMeasurementEngine();
		
		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_MEASUREMENT;
		OutBuffer[OutSize++] = 1;	//data bytes


		if (GlucoseMeter_StartMeasurement() == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}

		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;	
	}
}


void GluAbortMeasurementReq(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//execute command and send confirm
		GlucoseMeter_AbortMeasurement();
		
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_ABORT_MEASUREMENT;
		OutBuffer[OutSize++] = 0;	//data bytes
				
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		vfnTurnOffMeasurementEngine();
	}
}


void GluStartCalibrationReq(void)
{
	UINT16 knownGlucoseValue;

	if (ActualMeasurement == NO_MEASUREMENT)
	{
		vfnStartGLUMeasurementEngine();
		
		knownGlucoseValue = InBuffer[DATA_PACKET]<<8;
		knownGlucoseValue += InBuffer[DATA_PACKET+1];

		//execute command and send confirm
		OutBuffer[OutSize++] = CFM;
		OutBuffer[OutSize++] = GLU_START_CALIBRATION;
		OutBuffer[OutSize++] = 1;	//data bytes

		if (GlucoseMeter_StartCalibration(knownGlucoseValue) == TRUE)
		{
			OutBuffer[OutSize++] = ERROR_OK;
		}
		else
		{
			OutBuffer[OutSize++] = ERROR_BUSY;
		}
				
		(void)SerialComm_SendData();
		ActualMeasurement = GLU_MEASUREMENT;
	}
}


void GlucometerBloodDetectedInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send GLU_BLOOD_DETECTED indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_BLOOD_DETECTED;
		OutBuffer[OutSize++] = 0;				//data bytes
		(void)SerialComm_SendData();
	}			
}


void GlucometerMeasurementOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{		
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_MEASUREMENT_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue >> 8;			//high byte
		OutBuffer[OutSize++] = GlucoseMeter_GlucoseValue & 0x00FF;		//low byte
		(void)SerialComm_SendData();
		ActualMeasurement = NO_MEASUREMENT;
		
		vfnTurnOffMeasurementEngine();
	}
}


void GlucometerCalibrationOkInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_CALIBRATION_COMPLETE_OK;
		OutBuffer[OutSize++] = 2;	//data bytes: calibration constant
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant >> 8);		//high byte
		OutBuffer[OutSize++] = (UINT8)(GlucoseMeter_CalibrationConstant & 0x00FF);	//low byte
		(void)SerialComm_SendData();		
		ActualMeasurement = NO_MEASUREMENT;
		
		vfnTurnOffMeasurementEngine();
	}
}


void GlucometerDebugNewDataReadyInd(void)
{
	if (ActualMeasurement == GLU_MEASUREMENT)
	{
		UINT8 i;
		
		//Send  indication
		OutBuffer[OutSize++] = IND;
		OutBuffer[OutSize++] = GLU_DEBUG_MODE_NEW_DATA_READY;
		OutBuffer[OutSize++] = GLU_GRAPH_DATA_ARRAY_SIZE;	//data bytes: real time graph
		
		
		for (i=0; i<GLU_GRAPH_DATA_ARRAY_SIZE; i++)
		{			
			OutBuffer[OutSize++] = GlucoseMeter_GraphDataToPc[i];
		}
		(void)SerialComm_SendData();		
	}	
}


/*********************************
		Ecg
**********************************/		



void EcgHeartRateStartMeasurementReq(void)
{
	
}


void EcgHeartRateAbortMeasurementReq(void)
{

}



void EcgHeartRateMeasurementErrorInd(void)
{
	
}


//UINT8 Ecg_HeartRate = 0;

void EcgHeartRateMeasurementCompleteOkInd(void)
{
	
}


void EcgHeartBeatOccurredInd(void)
{
	
}


/*
	Ecg Dagnostic mode
*/

void EcgDiagnosticModeStartMeasurementReq(void)
{
  
	
}


void EcgDiagnosticModeStopMeasurementReq(void)
{
	
}





void EcgDiagnosticModeNewDataReadyInd(void)
{
	
}







/***************************************
		Bpm
****************************************/		

void BpmStartMeasurementReq(void)
{
	
}


void BpmAbortMeasurementReq(void)
{
	
}





void BpmMeasurementCompleteInd(void)
{
	
}



void BpmMeasurementErrorInd(void)
{
	
}



void BpmNewDataReadyInd(void)
{
	
}


void BmpSendPressureValue(void)
{
	
}



void BpmStartLeakTestReq(void)
{
	
}


void BpmAbortLeakTestReq(void)
{
	
}

void BpmLeakTestCompleteInd(void)
{
	
}





/* Tpm */
void HwtReadTemperatureReq(void)
{
	
}


void HwtReadHeightReq(void)
{
	
}

void HwtReadWeightReq(void)
{
	
}



void HwtTemperatureMeasurmentCompleteInd(void)
{
			
}


void HwtHeightMeasurementCompleteInd(void)
{
						
}


void HwtWeightMeasurementCompleteInd(void)
{
				
}





#define TIMER_SEND_SPIROMETER_GRAPH_INTERVAL	30		//ms

UINT16 Spirometer_Fet;
UINT16 Spirometer_Fvc;
UINT16 Spirometer_Fev;
UINT16 Spirometer_Fev1;
UINT16 Spirometer_Peft;
UINT16 Spirometer_Fiv;
UINT16 Spirometer_Fef;
UINT16 Spirometer_Fif;
UINT16 Spirometer_Pef;

/******************************************************************************
*                               Spirometer
*******************************************************************************/

void SprDiagnosticModeStartMeasurementReq(void)
{
	
}

void SprDiagnosticModeStopMeasurementReq(void)
{
	
}

void SprDiagnosticModeNewDataReadyInd(void)
{
	
}



/******************************************************************************
*                             Pulse Oximeter
*******************************************************************************/
/**
	Starts a pulse oximetry test (not diagnostic mode) and sends a CFM
*/
void PoxStartMeasurementReq(void)
{
	
}


/**
	Aborts a pulse oximetry test (not diagnostic mode) and sends a CFM
*/
void PoxAbortMeasurementReq(void)
{
	
}


/**
	When the POX measurement is complete, the MCU sends and IND to the host with the following data:
		- Oxygen saturation value
		- Heart rate
*/
void PoxMeasurementCompleteOkInd(void)
{
	
}


/**
	This IND is sent to the host if the POX measurement was not completed correctly
*/
void PoxMeasurementErrorInd(void)
{
	
}


/**
	Starts a pulse oximetry diagnostic test (graphical mode) and sends a CFM
*/
void PoxDiagnosticModeStartMeasurementReq(void)
{
     
}

/**
	Stops a pulse oximetry diagnostic test (graphical mode) and sends a CFM
*/
void PoxDiagnosticModeStopMeasurementReq(void)
{
	
}


/**
	This IND is sent to the host when new data from the POX graph is available.
	It sends the following:
		- 32 unsigned words containing the graph information
		- 1 byte containing current oxygen saturation value
		- 1 byte containing current heart rate value
*/
void PoxNewDataReadyInd(void)
{
	
}

/**
	This is for debugging purposes only.It sends both IR and Red graphs to the host
*/
void PoxDebugModeNewDataReadyInd(void)
{
	
}


#endif

/******************************************************************************
 *
 *    @name        USB_Notify_Callback
 *
 *    @brief       This function handles PSTN Sub Class callbacks
 *
 *    @param       controller_ID    : Controller ID
 *    @param       event_type       : PSTN Event Type
 *    @param       val              : gives the configuration value
 *
 *    @return      None
 *
 *****************************************************************************
 * This function handles USB_APP_CDC_CARRIER_ACTIVATED and
 * USB_APP_CDC_CARRIER_DEACTIVATED PSTN Events
 *****************************************************************************/

static void USB_Notify_Callback (
    uint_8 controller_ID,   /* [IN] Controller ID */
    uint_8 event_type,      /* [IN] PSTN Event Type */
    void* val               /* [IN] gives the configuration value */
)
{
    UNUSED (controller_ID)
    UNUSED (val)
    if(start_app == TRUE)
    {
        if(event_type == USB_APP_CDC_CARRIER_ACTIVATED)
        {
            start_transactions = TRUE;
        }
        else if(event_type == USB_APP_CDC_CARRIER_DEACTIVATED)
        {
            start_transactions = FALSE;
        }
    }
    return;
}

/******************************************************************************
 *
 *    @name        USB_App_Callback
 *
 *    @brief       This function handles Class callback
 *
 *    @param       controller_ID    : Controller ID
 *    @param       event_type       : Value of the event
 *    @param       val              : gives the configuration value
 *
 *    @return      None
 *
 *****************************************************************************
 * This function is called from the class layer whenever reset occurs or enum
 * is complete. After the enum is complete this function sets a variable so
 * that the application can start.
 * This function also receives DATA Send and RECEIVED Events
 *****************************************************************************/

static void USB_App_Callback (
    uint_8 controller_ID,   /* [IN] Controller ID */
    uint_8 event_type,      /* [IN] value of the event */
    void* val               /* [IN] gives the configuration value */
)
{
    UNUSED (controller_ID)
    UNUSED (val)
    if(event_type == USB_APP_BUS_RESET)
    {
        start_app=FALSE;
    }
    else if(event_type == USB_APP_ENUM_COMPLETE)
    {
        start_app=TRUE;
    }
    else if((event_type == USB_APP_DATA_RECEIVED))
    //&&            (start_transactions == TRUE))
    {
        /* Copy Received Data buffer to Application Buffer */
        USB_PACKET_SIZE BytesToBeCopied;
        APP_DATA_STRUCT* dp_rcv = (APP_DATA_STRUCT*)val;
        uint_8 index;
        BytesToBeCopied = (USB_PACKET_SIZE)((dp_rcv->data_size > DATA_BUFF_SIZE) ?
                                      DATA_BUFF_SIZE:dp_rcv->data_size);
        for(index = 0; index<BytesToBeCopied ; index++)
        {
            g_curr_recv_buf[index]= dp_rcv->data_ptr[index];
        }
        g_recv_size = index;
        (void)USB_Class_CDC_Interface_DIC_Recv_Data(CONTROLLER_ID, NULL, 0);
    }
    else if((event_type == USB_APP_SEND_COMPLETE)&&
                                              (start_transactions == TRUE))
    {
        /* Previous Send is complete. Queue next receive */
        (void)USB_Class_CDC_Interface_DIC_Recv_Data(CONTROLLER_ID, NULL, 0);
    }

    return;
}

/*****************************************************************************
*                        Measurement Engine Handler
*****************************************************************************/

void vfnTurnOffMeasurementEngine (void)
{
 vfnDisable_All_Operationals();
 DISABLE_DAC; 
}

void vfnStartECGMeasurementEngine (void)
{

}

void vfnStartPOXMeasurementEngine (void)
{

}




void vfnStartGLUMeasurementEngine (void)
{
#if (defined GLU_EXTERNAL_OPAMPS)
   vfnDisable_All_Operationals ();
#else
 opamp1_gp_mode() ;
 opamp2_gp_mode();
 TRIAMP1C0_HighMode();
 TRIAMP2C0_HighMode();
#endif
 GlucoseMeter_Init();
}

void vfnEnable_AFE (void)
{
  SIM_SCGC5 |= SIM_SCGC5_PORTC_MASK; //Clock to PORTC
  PORTC_PCR14 = PORT_PCR_MUX(1); //PTC14 as GPIO
  GPIOC_PDDR |= (1<<14); //PTC14 as output
  GPIOC_PCOR |= (1<<14); //AFEs Enabled
}